/**
 * Klasse Boid
 * 
 * 	Definiert die primitiven Elemente eines Schwarms, auch Boids genannt
 * 	Die hier abgelegten Attribute sind fr jedes Boid Objekt individuell,
 * 	Attribute die fr eine ganze Familie/ einen ganzen Schwarm gelten werden
 * 	in der "Oberklasse" BoidFamily abgelegt
 * 
 * Attribute:
 * 
 * 	boidID {int} eindeutiger Wert zur Identifikation eines Boid Objektes;
 * 	boidFamily {BoidFamily} Referenz auf die Familie / den Schwarm zu dem das Boid Objekt gehrt
 *	position {float[3]} momentaner Standort als Ortsvektor
 * 	direction {float[3]} momentane Richtng als normierter Richtungsvektor
 * 	speed {float} momentane Geschwindigkeit des Boid Objektes
 * 	isAlive {Boolean} definiert ob der Boid aktiv am Geschehen mitwirkt - true -> aktiv ; false -> inaktiv
 * 	neighbors {Boid[]} Referenzen auf im Sichtbereich gefundene Nachbarn des eigenen Schwarms
 * 	numNeighbors {int} Anzahl der momentanen Nachbarn im Sichtbereich
 * 	nearestNeighborDistance {float} Entfernung zum Nachbar mit derkrzesten Distanz zu diesem Boid
 * 	separators {Boid[]} Nachbarn die sich zu nahe an diesem Boid befinden (->Trennung/Separation)
 * 	separatorCount {int} Anzahl zu naher Nachbarn
 * 	gridCell {GridCell} Referenz auf die Zelle im Gitternetz in dem sich das Boid Objekt befindet
 * 	alienFamily {Boid[]} Referenzen auf gefundene (nicht feindliche) Boids anderer Schwre im Sichtbereich
 * 	alienFamilyCount {int} Anzahl (nicht feindlicher) Boids anderer Schwrme im Sichtbereich 
 * 	hostiles {Boid[]} Referenzen auf im Sichtbereich entdeckte feindliche Boids anderer Schwrme
 * 	hostileCount {int} Anzahl feindlicher Boids anderer Schwre im Sichtbereich
 * 	obstacles {Obstacle[]} Referenzen auf im Sichtbereich entdeckte Hindernisse
 * 	obstacleCount {int} Anzahl im Sichtbereich entdeckter Hindernisse
 * 	userControlled {Boolean} gibt an ob der Boid von manuell vom Benutzer gesteuert wird
 *  
 *
 */




/**
 * Boid(boidFamily) : Boid
 * 
 * 	Konstruktor und zugleich Definition der Klasse Boid
 * 
 * Parameter
 * 
 *  boidFamily {boidFamily} Verweis auf die Boidfamiliy /den Schwarm zu dem das Boid Objekt gehren soll
 * 
 * Rckgabewert
 * 
 * {Boid} neues Boid Objekt
 *
 */
function Boid(boidFamily) {
	this.boidID = boidID++; 
	this.boidFamily = boidFamily;
	this.position = [0,0,0];
	
	//initiale Richtung zufllig
	this.direction = vec3.normalize([Math.random()-0.5, Math.random()-0.5, Math.random()-0.5]);
	
	//Initiale Geschwindigkeit innerhalb der Mindest- und Maximalgeschwindigkeit der BoidFamily
	this.speed = Math.random() * (boidFamily.maxSpeed - boidFamily.minSpeed) + boidFamily.minSpeed;
	this.isAlive = true;
	this.neighbors = [];
	this.numNeighbors = 0;
	this.nearestNeighborDistance;
	this.separators = [];
	this.separatorCount = 0;
	this.gridCell;
	this.alienFamily = [];
	this.alienFamilyCount = 0;
	this.hostiles = [];
	this.hostileCount = 0;
	this.obstacles = [];
	this.obstacleCount = 0;
	this.userControlled = false;
	    
	    
	//Zuweisung des Startpunktes der eigenen "boidFamily"
	this.position[0] = this.boidFamily.birthPoint[0];
	this.position[1] = this.boidFamily.birthPoint[1];
	this.position[2] = this.boidFamily.birthPoint[2];
	    
};

/**
 * Methode generateNeighbors() : void
 * 
 * 	Erzeugen der Nachbarschaft eines friedlichen Boids
 * 
 * Klassenmethode zu Boid
 * 
 */
Boid.prototype.generateNeighbors = function(){
	//erkennt alle items im sichtfeld - begrenzt durch Sichtweite und ffnungswinkel
		
	var distance;
	var vecToItem;
	var angleToFound;
	
	//nachbarschaftsrelevante Counter zurcksetzen
	this.nearestNeighborDistance = -1;
	this.numNeighbors = 0; //hier wre der "Plan" bei jeder Iteration die Nachbarschaft neu zu generieren
	this.separatorCount = 0;
	this.alienFamilyCount = 0;
	this.hostileCount = 0;
	this.obstacleCount = 0;
	
	//ermitteln wie viele Zellen im Raster in jede Richtung betrachtet werden
	var searchLimit = Math.ceil(this.boidFamily.sightRange / theGrid.gridCellWidth); //Sichtradius auf nchste ganze Zahl aufgerundet
	
	//Suchbereich eingrenzen
	var searchstart = [];
	//untere Grenze der durchsuchten GridZellen
	vec3.subtract(this.gridCell.getGridLocation(), [searchLimit, searchLimit, searchLimit], searchstart);
	
	var searchend = [];
	//obere Grenze der durchsuchten GridZellen
	vec3.add(this.gridCell.getGridLocation(), [searchLimit, searchLimit, searchLimit], searchend);
	
	//Suchbereich validieren (innerhalb des zulssigen Grid-Bereichs)
	//d.h. nur Zellen betrachten die auch tatschlich im Grid existieren
	for(var i=0; i<3; i++){
		if(searchstart[i]<0){
			searchstart[i] = 0;
		}
		else if(searchstart[i]>=theGrid.gridLength[i]){
			searchstart[i] = theGrid.gridLength[i]-1;
		}
		
		if(searchend[i]<0){
			searchend[i] = 0;
		}
		else if(searchend[i]>=theGrid.gridLength[i]){
			searchend[i] = theGrid.gridLength[i]-1;
		}
	}
	
	var curCell; //lokaler Verweis auf momentan betrachtete Zelle im Grid
	
	for(var i=searchstart[0]; i<=searchend[0]; i++){
		for(var j=searchstart[1]; j<=searchend[1]; j++){
			for(var k=searchstart[2]; k<=searchend[2]; k++){
				
				//innerhalb einer Zelle
				curCell = theGrid.data[i][j][k];
				
				//eventuell vorhandene Hindernis in curCell in this.obstacles aufnehmen,
				//falls diese nicht schon dort vermerkt sind
				for(var l=0, maxl=curCell.obstacleCount; l<maxl;l++){
					var obstacleListed = false;
					var name = curCell.obstacles.name;
					for(var m=0,maxm=this.obstacleCount;m<maxm;m++){
						if(name == this.obstacles[m].name){
							obstacleListed = true; //Hindernis bereits bekannt
							break;
						}
					}
					if(!obstacleListed){ //nicht vermerktes Hindernis entdeckt
						this.obstacles[m] = curCell.obstacles[l];
						this.obstacleCount++;
					}
				}
				
				
				//Boids in "curCell" betrachten
				for(var l=0, maxl=curCell.itemCount; l<maxl; l++){
					
					//inaktive Boids und sich selbst NICHT betrachten
					if (!curCell.items[l].isAlive || this.boidID == curCell.items[l].boidID){
						continue;
					}
					
					//Abstand von diesem Objekt zum aktuell betrachteten bestimmen
					distance = itemDistance(this.position,curCell.items[l].position);
					
					//Objekt in Sichtweite
					if(distance<=this.boidFamily.sightRange){
										
						//Differenzvektor akuelles (this) Item und des in Reichweite gefundenen
						vecToItem = [curCell.items[l].position[0]-this.position[0], curCell.items[l].position[1]-this.position[1], curCell.items[l].position[2]-this.position[2]];
								
						//Winkel zwischen Richtungsvektor aktuelles Item (this) und dem vektor zur Position des gefundenen Item
						angleToFound = Math.acos(vec3.dot(this.direction,vecToItem)/(vec3.length(this.direction)*vec3.length(vecToItem)));
								
						//Item befindet sich innerhalb des Sichtbereichs
						if (angleToFound < degToRad(this.boidFamily.sightAngle)){
							
							//zuerst prfen ob item feindlich, danach ob es in einem fremdem Schwarm ist
							//ist dies der Fall wird die direkte Nachbarschaft nicht mehr betrachtet
							
							//feindlich (und nicht die eigene family) und in "hostileAvoidanceRange"
							if(curCell.items[l].boidFamily.name != this.boidFamily.name && curCell.items[l].boidFamily.isHostile && distance <= this.boidFamily.hostileAvoidanceRange){
								this.hostiles[this.hostileCount] = curCell.items[l];
								this.hostileCount++;
							}
							//bereits ein feind erkannt und aktuelles item ist keiner -> nichts zu tun
							else if(this.hostileCount > 0){
								continue;
							}
							//fremdes Schwarmmitglied in Sicht und entfernung kleiner "avoidanceRange"
							else if(curCell.items[l].boidFamily.name != this.boidFamily.name && distance <= this.boidFamily.alienFamilyAvoidanceRange){
								//this.alienFamilyInRange = true;
								this.alienFamily[this.alienFamilyCount] = curCell.items[l];
								this.alienFamilyCount++;
							}
							//bereits ein fremdes Schwarmmitglied erkannt und  erkannt und aktuelles item ist keiner -> nichts zu tun
							else if(this.alienFamilyCount > 0){
								continue;
							}
							//erst jetzt die eigenen Nachbarn
							
							else if(curCell.items[l].boidFamily.name == this.boidFamily.name){
								
								if(distance<=this.boidFamily.separationRange){
									this.separators[this.separatorCount] = curCell.items[l];
									this.separatorCount++;
								}
								else{
									this.neighbors[this.numNeighbors] = curCell.items[l];
									this.numNeighbors++;
									//distanz zum nhesten (gibts das Wort?) aktualisieren
									//aber nur wenns kein "separator" ist
									if(this.nearestNeighborDistance < 0 || distance < this.nearestNeighborDistance){
									this.nearestNeighborDistance = distance;
									}
								}
							}
							
							//hier wren dann nur noch die items der fremden Schwrme auerhalb der "avoidanceRange" brig
						}
					}		
				}
			}
		}
	}	
};
/**
 * Methode generateNeighborsForHostile() : void
 * 
 *	Erzeugen der Nachbarschaft eines feindseeligen Boids - werden getrennt behandelt, das unterschiedliches Verhalten unterstellt wird
 *
 * Klassenmethode zu Boid
 * 
 */
Boid.prototype.generateNeighborsForHostile = function(){
	//abgewandelte "generateNeighbors" fr feindselige Boids
		
	var distance;
	var vecToItem;
	var angleToFound;
	this.numNeighbors = 0; //Neighbors werden hier als "Opfer" zweckentfremdet
	this.separatorCount = 0;
	this.obstacleCount = 0;
	
	//ermitteln wie viele Zellen im Raster in jede Richtung betrachtet werden
	var searchLimit = Math.ceil(this.boidFamily.sightRange / theGrid.gridCellWidth); //Sichtradius auf nchste ganze Zahl aufgerundet
	
	//Suchbereich eingrenzen
	var searchstart = [];
	//untere Grenze der durchsuchten GridZellen
	vec3.subtract(this.gridCell.getGridLocation(), [searchLimit, searchLimit, searchLimit], searchstart);
	
	var searchend = [];
	//obere Grenze der durchsuchten GridZellen
	vec3.add(this.gridCell.getGridLocation(), [searchLimit, searchLimit, searchLimit], searchend);
	
	//Suchbereich validieren (innerhalb des zulssigen Grid-Bereichs)
	//d.h. nur Zellen betrachten die auch tatschlich im Grid existieren
	for(var i=0; i<3; i++){
		if(searchstart[i]<0){
			searchstart[i] = 0;
		}
		else if(searchstart[i]>=theGrid.gridLength[i]){
			searchstart[i] = theGrid.gridLength[i]-1;
		}
		
		if(searchend[i]<0){
			searchend[i] = 0;
		}
		else if(searchend[i]>=theGrid.gridLength[i]){
			searchend[i] = theGrid.gridLength[i]-1;
		}
	}
	
	var curCell;
	
	for(var i=searchstart[0]; i<=searchend[0]; i++){
		for(var j=searchstart[1]; j<=searchend[1]; j++){
			for(var k=searchstart[2]; k<=searchend[2]; k++){
				
				//innerhalb einer Zelle
				curCell = theGrid.data[i][j][k];
				
				for(var l=0, maxl=curCell.obstacleCount; l<maxl;l++){
					var obstacleListed = false;
					var name = curCell.obstacles.name;
					for(var m=0,maxm=this.obstacleCount;m<maxm;m++){
						if(name == this.obstacles[m].name){
							obstacleListed = true; //Hindernis bereits bekannt
							break;
						}
					}
					if(!obstacleListed){ //nicht vermerktes Hindernis entdeckt
						this.obstacles[m] = curCell.obstacles[l];
						this.obstacleCount++;
					}
				}
				
				
				for(var l=0, maxl=curCell.itemCount; l<maxl; l++){
					
					//inaktive Items und sich selbst NICHT betrachten
					if (!curCell.items[l].isAlive || this.boidID == curCell.items[l].boidID){
						continue;
					}
					distance = itemDistance(this.position,curCell.items[l].position);
						
					if(distance<=this.boidFamily.sightRange){
										
						//Differenzvektor akuelles (this) Item und des in Reichweite gefundenen
						vecToItem = [curCell.items[l].position[0]-this.position[0], curCell.items[l].position[1]-this.position[1], curCell.items[l].position[2]-this.position[2]];
						
						//Winkel zwischen Richtungsvektor aktuelles Item (this) und dem vektor zur Position des gefundenen Item
						angleToFound = Math.acos(vec3.dot(this.direction,vecToItem)/(vec3.length(this.direction)*vec3.length(vecToItem)));
								
						//Item befindet sich innerhalb des Sichtbereichs
						if (angleToFound < degToRad(this.boidFamily.sightAngle)){
							
							//bei "Jgern" beschrnken sich die Funktionen auf Separation und Opfer verfolgen
							
							//Artgenosse in Trennungsreichweite -> keine Jagd sondern Ausweichen
							if(curCell.items[l].boidFamily.name == this.boidFamily.name && distance<=this.boidFamily.separationRange){
								
								this.separators[this.separatorCount] = curCell.items[l];
								this.separatorCount++;
							}
							//bereits ein item zur Trennung vorhanden -> nichts zu tun
							else if(this.separatorCount > 0 || curCell.items[l].boidFamily.name == this.boidFamily.name){
								continue;
							}
							//keine Trennung -> Opfer verfolgen
							else /*if(curCell.items[l].boidFamily.name != this.boidFamily.name)*/{
								this.neighbors[this.numNeighbors] = curCell.items[l];
								this.numNeighbors++;
							}
						}
					}		
				}
			}
		}
	}	
};




/**
 * Methode alignDirection() : void
 * 
 * 	Angleichen der Richtung eines Boids an die seiner Nachbarn (gleiche BoidFamily),
 * 	entspricht dem als Ausrichten/Alignment bezeichneten Verhalten
 * 
 * Klassenmetode zu Boid
 */

Boid.prototype.alignDirection = function(){
	//Alignment
	
	//durchschnittliche Richtung errechnen und normieren
	var avgDir = [0.0,0.0,0.0];
	var avgSpeed = 0;
	
	//Addieren der Richtungsvektoren benachbarter Boids
	for(var i=0, max=this.numNeighbors; i<max; i++){
		avgDir[0] += this.neighbors[i].direction[0];
		avgDir[1] += this.neighbors[i].direction[1];
		avgDir[2] += this.neighbors[i].direction[2];
		
		avgSpeed += this.neighbors[i].speed;
	}

	
	//Summenvektor normieren - dadurch wird auch die Division durch die Anzahl der Nachbarn berflssig
	//gibt es keine Nachbarn wird die aktuelle Richtung beibehalten
	if(this.numNeighbors > 0){

		this.setDirection(vec3.normalize(avgDir));
		if(adjustSpeed)this.setSpeed(avgSpeed/this.numNeighbors,0.01);
		
		//experimentelle Geschwindigkeitsangleichung
	}
};


/**
 * Methode alignPosition() : void
 * 
 * 	Setzt die Richtung eines Boids auf die mittlere Position seiner Nachbarn zu,
 * 	entspricht dem als Cohesion/Zusammenhalt bezeichneten Verhalten
 * 
 * Klassenmethode zu Boid
 */

Boid.prototype.alignPosition = function(){
	//Cohesion
	
	//mittlere Richtung benachbarter Items wird als neue Richtung gesetzt
	
	var avgPos = [0.0,0.0,0.0];
	
	//Addieren der Positionsvektoren benachbarter Boids
	for(var i=0, max=this.numNeighbors; i<max; i++){
		avgPos[0] += this.neighbors[i].position[0];
		avgPos[1] += this.neighbors[i].position[1];
		avgPos[2] += this.neighbors[i].position[2];
	}
	
	//mittlere Position der Nachbarn bestimmen, Vektor zu diesem Punkt bestimmen und diesen
	//normiert als Richtungsvektor setzen
	//gibt es keine Nachbarn wird die aktuelle Richtung beibehalten
	if(this.numNeighbors > 0){
		avgPos[0] = avgPos[0] / this.numNeighbors;
		avgPos[1] = avgPos[1] / this.numNeighbors;
		avgPos[2] = avgPos[2] / this.numNeighbors;
		
		var newDir = [];
		vec3.subtract(avgPos,this.position,newDir);
		this.setDirection(vec3.normalize(newDir));
		if(adjustSpeed)this.changeSpeed(+0.01);
	}		
};

/**
 * Methode separate() : void
 * 
 * 	Setzt die Richtung eines Boids weg von seinen zu nahe geratenen Nachbarn,
 *  entspricht dem als Separation/Abstand halten bezeichneten Verhalten
 *  
 * Klassenmethode zu Boid
 * 
 */


Boid.prototype.separate = function(){
	//Separation
	
	//Fluchtrichtung weg von zu nahen Nachbarn ermitteln
	
	var sumNear = [0.0,0.0,0.0];
	
	//Addieren der Vektoren von zu nahen Boids zum aktuellen
	for(var i=0, max=this.numSeparators; i<max; i++){
		sumNear[0] += this.position[0] - this.separators[i].position[0];
		sumNear[1] += this.position[1] - this.separators[i].position[1];
		sumNear[2] += this.position[2] - this.separators[i].position[2];
	}
	
	//Fluchtrichtung zuweisen
	//gibt es keine zu nahen Nachbarn wird die aktuelle Richtung beibehalten
	if(this.separatorCount > 0){		
		this.setDirection(vec3.normalize(sumNear));
		if(adjustSpeed)this.changeSpeed(-0.01);
	}	
};

/**
 * Methode alienSeparate() : void
 * 
 * 	Setzt die Richtung eines Boids weg von zu nahe gekommenen Boids anderer Schwaerme,
 * 	funktioniert an sich genauso wie separate(), arbeitet aber auf einem anderen Datensatz
 * 
 * Klassenmethode zu Boid
 */

Boid.prototype.alienSeparate = function(){
	
	var sumNear = [0.0,0.0,0.0];
	
	//Addieren der Vektoren von zu nahen Boids zum aktuellen
	for(var i=0, max=this.alienFamilyCount; i<max; i++){
		sumNear[0] += this.position[0] - this.alienFamily[i].position[0];
		sumNear[1] += this.position[1] - this.alienFamily[i].position[1];
		sumNear[2] += this.position[2] - this.alienFamily[i].position[2];
	}

	//Fluchtrichtung zuweisen
	//gibt es keine zu nahen Nachbarn wird die aktuelle Richtung beibehalten
	if(this.alienFamilyCount > 0){		
		this.setDirection(vec3.normalize(sumNear));
		if(adjustSpeed)this.changeSpeed(-0.01);
	}
};
/**
 * Methode flee() : void
 * 
 * 	Setzt die Richtung eines Boids weg von zu nahe gekommenen Feinden,
 * 	funktioniert nlich wie separate, arbeitet aber auf einem anderen Datensatz und
 * 
 * Klassenmethode zu Boid
 * 
 */
Boid.prototype.flee = function(){
	
	var sumNear = [0.0,0.0,0.0];
	
	//Addieren der Vektoren von zu nahem Feinden zum aktuellen Boid
	for(var i=0, max=this.hostileCount; i<max; i++){
		sumNear[0] += this.position[0] - this.hostiles[i].position[0];
		sumNear[1] += this.position[1] - this.hostiles[i].position[1];
		sumNear[2] += this.position[2] - this.hostiles[i].position[2];
	}

	//Fluchtrichtung zuweisen
	//gibt es keine zu nahen Nachbarn wird die aktuelle Richtung beibehalten
	if(this.hostileCount > 0){		
		this.setDirection(vec3.normalize(sumNear));
		if(adjustSpeed)this.changeSpeed(+0.01);
	}
};



/**
 * Methode setDirection(targetDir)
 * 
 * 	Setzen einer neuen Richtung fr ein Boid Objekt - berschreitet diese
 * 	Richtungsnderung den maximalen Drehwinkel, wird zwar in die bergebene Richtung
 * 	gedreht, aber nur bis der maximale Drehwinkel erreicht ist
 * 
 * Parameter
 * 
 * 	targetDir {float[3]} die gewnschte neue Richtung
 * 
 * Klassenmethode zu Boid
 * 
 */

Boid.prototype.setDirection = function(targetDir){
	
	var target = vec3.normalize(targetDir);
	//Winkel zwischen aktellem Richtungsvektor und Zielrichtung - durch Skalarprodukt leider ohne Richtung
	var angleToVec = Math.acos(vec3.dot(this.direction, target));
	
	if(angleToVec > degToRad(this.boidFamily.maxRotation)){
		
		
		//Orthonormalvektor zwischen Richtungs- und Zielvektor
		//ergibt stets ein Rechtssystem - so dreht man also immer in die richtige Richtung
		var axis = [];
		vec3.cross(this.direction, target, axis);
		axis = vec3.normalize(axis);
		
		//Rotationsmatrix um normierte Drehachse axis mit maximalem Drehwinkel
		var rotM = getRotationMatrix(axis, this.boidFamily.maxRotation);
		
		//Rotationsmatrix auf Richtungsvektor anwenden (Matrixmultiplikation)
		var newDirection = [];
		newDirection[0] = rotM[0]*this.direction[0] + rotM[1]*this.direction[1] + rotM[2]*this.direction[2]; 
		newDirection[1] = rotM[3]*this.direction[0] + rotM[4]*this.direction[1] + rotM[5]*this.direction[2];
		newDirection[2] = rotM[6]*this.direction[0] + rotM[7]*this.direction[1] + rotM[8]*this.direction[2];
		
		this.direction = vec3.normalize(newDirection);
	}
	else{
		//Drehwinkel ist kleiner als maxRotation - Drehung kann also ohne Einschrnkung vollzogen werden
		this.direction =target;
	}
	//Workaround zur Vermeidung von "echten" Nullkomponenten, die ansonsten zu kaum beherrschbaren
	//Seiteneffekten fhren - Boids mit einem Nullvektor als Richtung lassen sich nur schlecht zeichnen
	
	
	if(this.direction[0]==0){
		this.direction[0]=0.0001;
	}
	if(this.direction[1]==0){
		this.direction[1]=0.0001;
	}
	if(this.direction[2]==0){
		this.direction[2]=0.0001;
	}
};

/**
 * Methode reactivate(): void
 * 
 * 	Reaktiviert ein inaktives Boid Objekt indem es seine Attribute zuruecksetzt, genauer werden
 * 	nur Position, Richtung und Geschwindigkeit neu zugewiesen - Nachbarschaft und dergleichen werden ohnehin
 * 	beim Aufruf von generateNeighbors() zurckgesetzt.
 * 	Die Zugehrigkeit zu einer BoidFamily wird nicht verndert.
 * 	Tieferer Zweck dieser Methode ist die Vermeidung unntig erzeugter Objekte indem man "alte recycled"
 * 
 * Klassenmethode zu Boid
 * 
 */
Boid.prototype.reactivate = function(){
	
	this.position[0] = this.boidFamily.birthPoint[0]; this.position[1] = this.boidFamily.birthPoint[1]; this.position[2] = this.boidFamily.birthPoint[2];
	this.direction = vec3.normalize([Math.random()-0.5,Math.random()-0.5,Math.random()-0.5]);
	this.speed = Math.random() * (this.boidFamily.maxSpeed - this.boidFamily.minSpeed) + this.boidFamily.minSpeed;;
	this.isAlive = true;
};


/**
 * Methode Boolean reflectBorders() : Boolean
 * 
 * 	Reflektiert das Boidobjekt an der Begrenzung - d.h. wrde eine Komponente des Positionsvektors in der nchsten Iteration die
 *	Grenzen berschreiten, wird diese Komponente im Richtungsvektor negiert.
 * 
 * Rckgabewert
 * 	{Boolean} wurde eine Reflexion durchgefhrt?
 * 
 * Klassenmethode zu Boid
 * 
 */
Boid.prototype.reflectBorders = function (){
	
	
    var near = theGrid.gridBorderNear;
    var far = theGrid.gridBorderFar;
    
    
	var reflection = false;
	if(this.position[0] + this.speed*this.direction[0] > near[0] || this.position[0] + this.speed*this.direction[0] < far[0]){
		this.direction[0] = -this.direction[0];
		reflection = true;
	}
	if(this.position[1] + this.speed*this.direction[1] > near[1] || this.position[1] + this.speed*this.direction[1] < far[1]){
		this.direction[1] = -this.direction[1];
		reflection = true;
	}
	if(this.position[2] + this.speed*this.direction[2] > near[2] || this.position[2] + this.speed*this.direction[2] < far[2]){
		this.direction[2] = -this.direction[2];
		reflection = true;
	}
	return reflection;
};

/**
 * Methode smoothBorders(softBorder) : Boolean
 * 
 * 	Gert ein Boid Objekt in einer Komponente zu nah an die begrenzende "Wand" wird von dieser weg gesteuert.
 * 
 * Parameter
 * 
 * 	softBorder {float} Abstand bei dessen Unterschreitung der Boid "zu nah" ist
 * 
 * Rckgabewert
 * 
 * 	{Boolean} Wurde ein Ausweichvorgang durchgefhrt?
 * 
 * Klassenmethode zu Boid
 * 
 */
Boid.prototype.smoothBorders = function(softBorder){
	var alternate = [0,0,0];
	var evade = false;
	for(var i=0; i<3; i++){
		if(this.position[i] < (theGrid.gridBorderFar[i] + softBorder)){
			alternate[i] = 1;
			evade = true;
		}
		else if(this.position[i] > (theGrid.gridBorderNear[i] - softBorder)){
			alternate[i] = -1;
			evade = true;
		}
	}
	if(evade){
		this.setDirection(alternate);
		if(adjustSpeed)this.changeSpeed(-0.01);
	}
	return evade;
};
/**
 * Methode changeSpeed(value) : void
 * 
 * 	nderung der Geschwindigkeit eines Boids um value - stellt sicher, dass sich die neues Geschwindigkeit
 * 	innerhalb des Intervalls [this.boidFamily.minSpeed ; this.boidFamily.maxSpeed ] befindet
 * 
 * Parameter
 * 
 * 	value {float} vorzeichenbehaftete Geschwindigkeitsnderung
 * 
 * Klassenmethode zu Boid
 */
Boid.prototype.changeSpeed = function(value){
	var newSpeed = this.speed + value;
	if(newSpeed > this.boidFamily.minSpeed && newSpeed < this.boidFamily.maxSpeed){
		this.speed = newSpeed;
	}
};

/**
 * Methode setSpeed(newSpeed, ratio) : void
 * 
 * 	ndert die Geschwindigkeit eines Boids in Richtung newSpeed aber maximal um ratio - stellt sicher,
 *  dass sich die neues Geschwindigkeit	innerhalb des 
 *  Intervalls [this.boidFamily.minSpeed ; this.boidFamily.maxSpeed ] befindet
 *  
 * Parameter
 *  
 *  newSpeed {float} gewnschte neue absolute Geschwindigkeit
 * 	ratio {float}  maximale Geschwindigkeitsnderung
 * 
 * Klassenmethode zu Boid
 * 
 */
Boid.prototype.setSpeed = function(newSpeed, ratio){
	if(newSpeed > this.boidFamily.minSpeed && newSpeed < this.boidFamily.maxSpeed){
		if(Math.abs(this.speed - newSpeed) < ratio){
			this.speed = newSpeed;
		}
		else{
			if(newSpeed < this.speed){
				this.speed -=ratio;
			}
			else{
				this.speed +=ratio;
			}
		}
	}
};

/**
 * Methode avoidObstacles() : void
 * 
 * 	Prft bei eventuell im Sichtbereich des Boids gefundenen Hindernissen, ob diesem Ausgewichen werden muss 
 * 
 * Rckgabewert
 * 
 * {Boolean} wurde eine Ausweichrichtung gesetzt?
 * 
 * Klassenmethode zu Boid
 * 
 */
Boid.prototype.avoidObstacles = function(){
	for(var i = 0, max = this.obstacleCount; i<max;i++){
		//hier prfen ob Ausweichen ntig
		if(itemDistance([this.position[0],this.position[1], 0],
				[this.obstacles[i].location[0],this.obstacles[i].location[1],0])
				<= this.obstacles[i].radius + this.boidFamily.obstacleAvoidanceRange ){
			//Boid befindet sich im in der hhe unbegrenzen Hllzylinder
			if(this.position[2] >= this.obstacles[i].location[2] - this.boidFamily.obstacleAvoidanceRange &&
			   this.position[2] <= this.obstacles[i].location[2] + this.obstacles[i].height + this.boidFamily.obstacleAvoidanceRange){
				// Boid innerhalb des Hllzylinders -> Fluchtrichtung zuweisen
				if(this.position[2] <= this.obstacles[i].location[2]){
					//unterhalb des Zylinders
					this.setDirection([this.position[0] - this.obstacles[i].location[0],
					                   this.position[1] - this.obstacles[i].location[1],
					                   this.position[2] - this.obstacles[i].location[2]
										]);
					if(adjustSpeed)this.changeSpeed(-0.01);
					return true;
				}
				else if(this.position[2] >= this.obstacles[i].location[2] + this.obstacles[i].height){
					//oberhalb des Zylinders
					this.setDirection([this.position[0] - this.obstacles[i].location[0],
					                   this.position[1] - this.obstacles[i].location[1],
					                   this.position[2] - this.obstacles[i].location[2] - this.obstacles[i].height
										]);
					if(adjustSpeed)this.changeSpeed(-0.01);
					return true;
				}
				else {
					//seitlich des Zylinders 
					this.setDirection([this.position[0] - this.obstacles[i].location[0],
					                   this.position[1] - this.obstacles[i].location[1],
					                   0
										]);
					if(adjustSpeed)this.changeSpeed(-0.01);
					return true;
				}
			}
		}
		
	}
	return false;
};

/**
 * Methode checkCollision() : void
 * 
 * 	Prfung auf eine Kollision eines Jgers mit einem Opfer - findet eine solche Kollision statt,
 * 	hat der Jger seine "Beute" erwischt und vernichtet diese
 * 
 * Klassenmethode zu Boid
 * 
 */
Boid.prototype.checkCollision = function(){
	for(var i=0,max = this.numNeighbors; i<max;i++){
		if(itemDistance(this.position, this.neighbors[i].position)< this.boidFamily.collisionRadius + this.neighbors[i].boidFamily.collisionRadius){
			//Beute erlegt
			this.neighbors[i].isAlive = false;
			
			//Beute wird verdaut, also Vollbremsung
			if(adjustSpeed)this.speed = this.boidFamily.minSpeed;
		}
	}
};

/**
 * Methode userControl() : void
 * 
 * 	Wird zur direkten Steuerung eines Boids durch den Benutzer genutzt
 * 
 * Klassenmethode zu Boid
 * 
 */
Boid.prototype.userControlled = function(){
	
};


/**
 * Methode draw() : void 
 * 
 * 	Transformiert das Koordinatensystem um ein Boid-Objekt zu zeichnen, gibt zum zeichnen Anweisung und berechnet aus
 * 	Position und Richtung die neue Position fr die nchste Iteration
 * 
 * Parameter
 * 
 * 	bodyTexture {int} Index der Textur mit der der Krper des Boids berzogen werden soll
 * 	headTexture {int} Index der Textur mit der der Kopf des Boids berzogen werden soll
 * 
 * Klassenmethode zu Boid
 * 
 */
Boid.prototype.draw = function (bodyTexture, headTexture) {
    mvPushMatrix();
    
    //Verschieben auf den Punkt auf dem spter der Boid gezeichnet werden soll
    mat4.translate(mvMatrix, this.position);
    
    //um das Koordinatensystem korrekt an der Richtung auszurichten mssen
    //zunchst die Winkel zur Drehung der Achsen mit Hilfe von Kugelkoordinaten bestimmt werden 

    var theta = Math.acos(this.direction[2]/*/vec3.length(this.direction)*/); //arccos(z/|r|)
    
    //vec3.length(this.direction) sollte eigentlich 1 sein
    var phi;

    if (this.direction[0] > 0){
		phi = Math.atan(this.direction[1]/this.direction[0]); //arctan(y/x)
	}
    else if (this.direction[0] == 0){
		phi = sgn(this.direction[1])*Math.PI/2; //sgn(y) * pi/2
	}
    else if (this.direction[0] < 0 && this.direction[1] >= 0){
		phi = Math.atan(this.direction[1]/this.direction[0]) + Math.PI; //arctan(y/x) + pi
	}
    else if (this.direction[0] < 0 && this.direction[1] < 0){
	//hier sollte es auch n normales else tun
		phi = Math.atan(this.direction[1]/this.direction[0]) - Math.PI; //arctan(y/x) - pi
	}
	
    mat4.rotate(mvMatrix, phi, [0.0,0.0,1.0]);
    mat4.rotate(mvMatrix, -(Math.PI/2 - theta),[0.0,1.0,0.0]);
    
    //Vektor von aktueller zu neuer Position bestimmen - also Richtung * Geschwindigkeit
    var scaled = [];
    scaled[0] = this.direction[0] * this.speed;
    scaled[1] = this.direction[1] * this.speed;
    scaled[2] = this.direction[2] * this.speed;
	
     
    //zeichnen
    models.basic(bodyTexture, headTexture);
	
    //neue Position speichern
    vec3.add(this.position, scaled, this.position);

    mvPopMatrix();
};
